//
// Sunflow Exporter.js
//
// v.20090515
//  required version : Cheetah3D v3.5
//
// 20070117 first.
// 20070124 alpha.
// 20070127 public.
// 20070130 fixed normal culculation bug.
// 20070205 fixed texture flip bug, giometry bug.
// 20070209 add hair curve future from Hair.js script.
// 20090514 fixed some bug
// 20090515 fixed bug when bump material
//
// @author Hiroto Tsubaki (tg@tres-graficos.jp)
// special thanks to Martin, Peter, and Yuichi Masuda. thanks to developers of Sunflow Renderer.
//
// Description: Export Scene for Sunflow (.sc file).
// Usage: Place this into ~/Library/Application Support/Cheetah3D/scripts/Tool folder. restart Cheetah3D, then select from Tools -> Script -> Tool Script
//
// <- Usage Memo ->
// Rendering.
// java -server -Xmx[Memory Size]m -jar [path/to/sunflow.jar] -threads [thread_count] -o /path/to/result_image.[png/hdi/tga]
// Texture Baking.
// java -server -Xmx[Memory Size]m -jar [path/to/sunflow.jar] -bake [object_name].instance -bakedir [ortho/view] -threads [thread_count] -o /path/to/baked_image.[png/hdi/tga]
//
// this from Todd's script
// add function to Array class.
if( !Array.indexOf) {
    Array.prototype.indexOf = function(str) {
        for(var i = 0; i < this.length; i++) {
            if(this[i] == str)
                return i;
        }
        return -1;
    }
}
//
var SC_FILE; // File object.
var SC_PATH = '';

var sunflow_ver = '0.07.2';
var sunflow_cmd = 'java -server -Xmx1024m -jar /Applications/sunflow-bin-v0.07.1/sunflow.jar';

var SHAPE_COUNT = 0;
var MAT_COUNT = 0;
var MOD_COUNT = 0;
var LIGHT_COUNT = 0;

var DEF_MAT_NAME = 'CM_def';

var PRIMITIVE_LIST = new Array; // store primitive (Ball, Cube .etc.)
var MESH_LIST = new Array; // store polygon object
var LIGHT_LIST = new Array; // store lights
var MAT_LIST = new Array; // store materials

var SUN_LIGHT = undefined;
var BAKED_OBJ = undefined;

var Normal_List = new Array(new Vec3D(1, 0, 0), new Vec3D(0, 1, 0), new Vec3D(0, 0, 1));

var Accel_list = new Array("none","kdtree","uniformgrid","null","bih");
var Bucket_list = new Array("row","column","diagonal","spiral","hilbert");
var ImageFilter_list = new Array("box","gaussian","mitchell","triangle","catmull-rom","blackman-harris","sinc","lanczos");

var GI_Types = new Array("ambocc","igi","fake","path","irr-cache","irr-cahce with global photon");
var Camera_Types = new Array("no override","pinhole","thinlens","spherical","fisheye");
var Photon_Types = new Array("caustics","global");

function buildUI(tool) {
    tool.addParameterSeparator("Sunflow Exporter");
    
    tool.addParameterBool("separate mesh file",0,0,1,true,true);
    tool.addParameterBool("use primitive",1,0,1,true,true);
    
    tool.addParameterSeparator("Trace Depths");
    tool.addParameterInt("diffuse",3,1,16,true,true);
    tool.addParameterInt("reflection",2,1,16,true,true);
    tool.addParameterInt("refraction",2,1,16,true,true);
    
    tool.addParameterSeparator("Camera Override");
    tool.addParameterSelector("camera type",Camera_Types,true,true);
    tool.addParameterInt("lens sides",6,0,60,true,true);
    tool.addParameterFloat("lens rotation",30,0,360,true,true);
    
    tool.addParameterSeparator("AA");
    tool.addParameterInt("Min AA",0,-4,5,true,true);
    tool.addParameterInt("Max AA",2,-4,5,true,true);
    tool.addParameterSelector("Image Filter",ImageFilter_list,true,true);
    tool.addParameterInt("AA samples",1,1,128,true,true);
    tool.addParameterBool("AA jitter",0,0,1,true,true);
    
    tool.addParameterSeparator("Accelerators");
    tool.addParameterSelector("Accelerator Type",Accel_list,true,true);
    tool.addParameterSelector("Bucket Order",Bucket_list,true,true);
    tool.addParameterInt("Bucket Size",32,8,64,true,true);
    
    tool.addParameterSeparator("Photon");
    tool.addParameterBool("photon",0,0,1,true,true);
    tool.addParameterSelector("photon type",Photon_Types,true,true);
    tool.addParameterInt("photon count",1000000,0,100000000,true,true);
    
    tool.addParameterSeparator("Global Illumination");
    tool.addParameterSelector("GI Type",GI_Types,true,true);
    
    tool.addParameterSeparator("[ambocc] Settings");
    tool.addParameterFloat("Bright (R)",0.1,0,1,true,true);
    tool.addParameterFloat("Bright (G)",0.1,0,1,true,true);
    tool.addParameterFloat("Bright (B)",0.1,0,1,true,true);
    tool.addParameterFloat("Dark (R)",0.01,0,1,true,true);
    tool.addParameterFloat("Dark (G)",0.01,0,1,true,true);
    tool.addParameterFloat("Dark (B)",0.01,0,1,true,true);
    tool.addParameterFloat("Distance",0.1,0,150,true,true);
    
    tool.addParameterSeparator("HDRI - IBL");
    tool.addParameterInt("IBL Samples",2,0,1024,true,true);

    tool.addParameterSeparator("Physical Sky - [ distant light as sun ]");
    tool.addParameterInt("sun samples",4,2,256,true,true);
    tool.addParameterFloat("turbidity",4,0,15,true,true);
    
    tool.addParameterSeparator("Export");
    tool.addParameterButton("Export","export","exportSC");
    //tool.addParameterSeparator("Export & Run");
    //tool.addParameterButton("Export & Run","export & run","exportSCAndRun");
}

function shapeName(name) {
    var names = name.split(" ").join("");
    SHAPE_COUNT += 1;
    return "CH_"+names+"_"+SHAPE_COUNT;
}

function matName(type, name) {
    var names = name.split(" ").join("");
    MAT_COUNT += 1;
    return "CM_"+type+"_"+names+"_"+MAT_COUNT;
}

function modName(type, name) {
    var names = name.split(" ").join("");
    MOD_COUNT += 1;
    return "CMd_"+type+"_"+names+"_"+MOD_COUNT;
}
function lightName(name) {
    var names = name.split(" ").join("");
    LIGHT_COUNT += 1;
    return "CL_"+names+"_"+LIGHT_COUNT;
}

function filenameForTexture(filepath) {
    if (filepath == "none") return false;
    var file = new File(filepath);
    return file.lastPathComponent();
}

function tagOfObject(obj, tagType) {
    var i;
    var tagCount = obj.tagCount();
    var hasTag = false;
    for (i = 0;i < tagCount;i++) {
        var tag = obj.tagAtIndex(i);
        if (tag.type() == tagType) {
            hasTag = true;
            returnTag = tag;
        }
    }
    return (hasTag)? returnTag : false;
}

function vector3Line(vec) {
    return vec.x.toPrecision(6)+" "+vec.y.toPrecision(6)+" "+vec.z.toPrecision(6);
}

function colorLine(vec) {
    return "{ \"sRGB nonlinear\" "+vector3Line(vec)+" }";
}
function vector4Line(vec) {
    return vec.r.toPrecision(6)+" "+vec.g.toPrecision(6)+" "+vec.b.toPrecision(6)+" "+vec.a.toPrecision(6);
}

function storeObjects(obj, rot, scale, tool) {
    var obj_rot = obj.getParameter("rotation");
    var obj_scale = obj.getParameter("scale");
    var use_prim = tool.getParameter("use primitive");
    //var use_csg = tool.getParameter("use CSG");
    //
    //use_prim = false;
    use_csg = false;
    //
    rot = rot.add(obj_rot);
    scale = new Vec3D(obj_scale.x*scale.x, obj_scale.y*scale.y, obj_scale.z*scale.z);
    //
    if (obj.family() == NGONFAMILY) {
        if (obj.type() == BALL && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == PLANE && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == TORUS && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == CONE && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else { // Polygon object or other Prametrics
            MESH_LIST.push(new Array(obj, rot, scale));
        }
    } else if (obj.family() == LIGHTFAMILY && obj.type() != 55) { // light family, but not sunlight
        //printParameterInfo(obj);
        if (obj.getParameter("lightType") == 2) SUN_LIGHT = obj; // treating distant light as sun. 
        else LIGHT_LIST.push(new Array(obj, rot, scale));
    }
    //
    if (obj.isCeatorObj() == false) {
        var i = 0;
        var childCount = obj.childCount();
        for (i = 0;i < childCount;i++) {
            storeObjects(obj.childAtIndex(i), rot, scale, tool);
        }
    }
}

function writeCaption(str) {
    SC_FILE.writeln("## -----  "+str+"\n");
}

function exportSCAndRun(tool) {
    exportSC(tool);
    path = sunflow_cmd+" "+SC_PATH;
    OS.system(path);
}

function exportSC(tool) {
    var doc = tool.document();
    // get path for SCN file
    var path = OS.runSavePanel("sc");
    if (path == null) {
        return;
    }
    SC_PATH = path;
    
    PRIMITIVE_LIST = new Array();
    MESH_LIST = new Array();
    //
    LIGHT_LIST = new Array();
    MAT_LIST = new Array();
    SUN_LIGHT = undefined;
    BAKED_OBJ = undefined;
    
    // initialize counts;
    SHAPE_COUNT = 0;
    MAT_COUNT = 0;
    MOD_COUNT = 0;
    LIGHT_COUNT = 0;
    //
    print("---- Sunflow Export.js ----");
    SC_FILE = new File(path);
    SC_FILE.open(WRITE_MODE);
    //
    writeCaption("sunflow scene description language file\n##"+" -----  for Sunflow v."+sunflow_ver+"\n## -----  created by Cheetah3D");

    // storing
    // object, mesh, light
    storeObjects(doc.root(), new Vec3D(0,0,0), new Vec3D(1,1,1), tool);
    
    writeCaption("settings");
    writeSetting(tool);
    // material
    writeCaption("shaders");
    SC_FILE.writeln("shader {\n\tname \""+DEF_MAT_NAME+"\"\n\ttype diffuse\n\tdiff 1 1 1\n}");
    var matCount = doc.materialCount();
    for (i = 0;i < matCount;i++) {
        var material = doc.materialAtIndex(i);
        var mat_name = writeMaterial(material, [""], "block");
        MAT_LIST.push(mat_name);
    }
    //
    writeCaption("camera and sky");
    writeCamera(tool);
    // lights
    var lightCount = LIGHT_LIST.length;
    if (lightCount) writeCaption("lights");
    for (i = 0;i < lightCount;i++) {
        writeLight(LIGHT_LIST[i], tool);
    }
    // primitive
    var primCount = PRIMITIVE_LIST.length;
    if (primCount) writeCaption("primitives");
    for (i = 0;i < primCount;i++) {
        writePrimitive(PRIMITIVE_LIST[i], tool);
    }
    // mesh (polygon object)
    var meshCount = MESH_LIST.length;
    if (meshCount) writeCaption("mesh");
    for (i = 0;i < meshCount;i++) {
        writeMesh(MESH_LIST[i], tool);
    }
    
    SC_FILE.close();
}

function writeCamera(tool) {
    var doc = tool.document();
    var cam = doc.activeCamera();
    var cam_tagDof = tagOfObject(cam,DOFTAG);
    var cam_tagHDRI = tagOfObject(cam,HDRITAG);
    var cam_tagGI = tagOfObject(cam,RADIOSITYTAG);
    var cam_resX = cam.getParameter("resolutionX");
    var cam_resY = cam.getParameter("resolutionY");
    var cam_projection = cam.getParameter("projection");
    var cam_mat = cam.obj2WorldMatrix();
    var cam_oType = tool.getParameter("camera type");
    var cam_rot = cam.getParameter("rotation");
    var fov = cam.getParameter("fieldOfView");
    var cam_distance = 1;
    if (cam_tagDof != false) {
        cam_distance = cam_tagDof.getParameter("focalDistance");
    } else {
        cam_distance = cam.getParameter("distance");
    }

    var eyeV = new Vec4D(cam_mat.m03, cam_mat.m13, cam_mat.m23, cam_mat.m33);
    var targetV = new Vec4D(0,0,cam_distance * -1,1);
    var upV = new Vec4D(0,1,0,0);
    var cam_matRot = new Mat4D(ROTATE_HPB, cam_rot.x, cam_rot.y, cam_rot.z);
    
    targetV = cam_mat.multiply(targetV);
    upV = cam_mat.multiply(upV);
    
    var cam_type = 'pinhole';
    if (cam_tagDof != false) cam_type = 'thinlens';
    if (cam_projection == 1) cam_type = 'spherical';
    if (cam_oType != 0) {
        cam_type = Camera_Types[cam_oType];
        if (cam_type == 'thinlens' && ! cam_tagDof) cam_type = 'pinhole'; // error check
    }
    
    var cam_aspect = cam_resX / cam_resY;
    fov = fov * cam_aspect;
    
    SC_FILE.writeln("\ncamera {");
    SC_FILE.writeln("\ttype "+cam_type);
    SC_FILE.writeln("\teye "+vector3Line(eyeV));
    SC_FILE.writeln("\ttarget "+vector3Line(targetV));
    SC_FILE.writeln("\tup "+vector3Line(upV));
    if (cam_type != 'spherical' && cam_type != 'fisheye') SC_FILE.writeln("\tfov "+fov.toPrecision(3));
    if (cam_type != 'spherical' && cam_type != 'fisheye') SC_FILE.writeln("\taspect "+cam_aspect.toPrecision(3));
    //printParameterInfo(cam_tagDof);
    if (cam_tagDof) {
        SC_FILE.writeln("\tfdist "+cam_tagDof.getParameter("focalDistance"));
        SC_FILE.writeln("\tlensr "+cam_tagDof.getParameter("lensRadius"));
        if (cam_oType == 2) {
            SC_FILE.writeln("\tsides "+tool.getParameter("lens sides"));
            SC_FILE.writeln("\trotation "+tool.getParameter("lens rotation"));
        }
    }
    SC_FILE.writeln("}\n");
    
    var hasSun = false;
    // sky setting
    if (SUN_LIGHT) {
        hasSun = true;
        var sun_mat = SUN_LIGHT.obj2WorldMatrix();
        var pos = new Vec3D(sun_mat.m03, sun_mat.m13, sun_mat.m23);
        var sun_samples = tool.getParameter("sun samples");
        var turbidity = tool.getParameter("turbidity");
        if (pos.y > 0) {
            SC_FILE.writeln("\nlight {");
            SC_FILE.writeln("\ttype sunsky");
            SC_FILE.writeln("\tup 0 1 0");
            SC_FILE.writeln("\teast 0 0 1");            
            SC_FILE.writeln("\tsundir "+vector3Line(pos));
            if (turbidity > 0) SC_FILE.writeln("\tturbidity "+turbidity.toPrecision(6));
            if (sun_samples > 0) SC_FILE.writeln("\tsamples "+sun_samples);
            SC_FILE.writeln("}\n");
        } else {
            hasSun = false;
        }
    }
    if (! hasSun) {
        var cam_bgImage = '';
        if (cam_tagHDRI != false && cam_tagHDRI.getParameter("hdriBackgroundImage") != "none") {
            var hdri = cam_tagHDRI.getParameter("hdriBackgroundImage");
            var ibl_samples = tool.getParameter("IBL Samples");
            cam_bgImage = hdri; //filenameForTexture(hdri);
        }
        
        if (cam_bgImage) {
            SC_FILE.writeln("\nlight {");
            SC_FILE.writeln("\ttype ibl");
            SC_FILE.writeln("\timage \""+cam_bgImage+"\"");
            SC_FILE.writeln("\tcenter 0 0 1");
            SC_FILE.writeln("\tup 0 1 0");
            SC_FILE.writeln("\tlock true\n");
            SC_FILE.writeln("\tsamples "+ibl_samples);
            SC_FILE.writeln("}\n");
        }
        if ( ! cam_bgImage || ! cam_tagHDRI.getParameter("hdriPrimBackground")) {
            var bg_color = cam.getParameter("backgroundColor");
            // constant sky
            SC_FILE.writeln("background {");
            SC_FILE.writeln("\tcolor "+colorLine(bg_color));
            SC_FILE.writeln("}");
        }
    }
    
    var photonSwc = tool.getParameter("photon");
    
    if (cam_tagGI) {
        var gi_type = GI_Types[tool.getParameter("GI Type")];
        var gi_samples = cam_tagGI.getParameter("samples");
        var gi_min = cam_tagGI.getParameter("minDist") / 5;
        var gi_max = cam_tagGI.getParameter("maxDist") / 5;
        var gi_error = cam_tagGI.getParameter("error") / 10;
        SC_FILE.writeln("gi {");
        if (gi_type == 'irr-cahce with global photon') gi_typeName = 'irr_cache';
        else gi_typeName = gi_type;
        SC_FILE.writeln("\ttype "+gi_typeName);
        switch(gi_type) {
            case 'ambocc':
                SC_FILE.writeln("\tbright "+colorLine(new Vec3D(tool.getParameter("Bright (R)"), tool.getParameter("Bright (G)"), tool.getParameter("Bright (B)"))));
                SC_FILE.writeln("\tdark "+colorLine(new Vec3D(tool.getParameter("Dark (R)"), tool.getParameter("Dark (G)"), tool.getParameter("Dark (B)"))));
                SC_FILE.writeln("\tsamples "+gi_samples);
                SC_FILE.writeln("\tmaxdist "+tool.getParameter("Distance"));
                break;
            case 'igi':
                SC_FILE.writeln("\tsamples 64");
                SC_FILE.writeln("\tsets 1");
                SC_FILE.writeln("\tb 0.0003");
                SC_FILE.writeln("\tbias-samples 0");
                if (!photonSwc) OS.messageBox("this gi setting requires photons.","some gi settings requires photons. Exporter adds photons automatically.");
                photonSwc = 1;
                break;
            case 'fake':
                SC_FILE.writeln("\tup "+vector3Line(new Vec3D(0, 1, 0)));
                SC_FILE.writeln("\tsky "+colorLine(new Vec3D(tool.getParameter("Bright (R)"), tool.getParameter("Bright (G)"), tool.getParameter("Bright (B)"))));
                SC_FILE.writeln("\tground "+colorLine(new Vec3D(tool.getParameter("Dark (R)"), tool.getParameter("Dark (G)"), tool.getParameter("Dark (B)"))));
               break;
            case 'path':
                SC_FILE.writeln("\tsamples "+gi_samples);
                break;
            case 'irr-cache':
                SC_FILE.writeln("\tsamples "+gi_samples);
                SC_FILE.writeln("\ttolerance "+gi_error);
                SC_FILE.writeln("\tspacing "+gi_min+" "+gi_max);
                if (!photonSwc) OS.messageBox("this gi setting requires photons.","some gi settings requires photons. Exporter adds photons automatically.");
                photonSwc = 1;
                break;
            case 'irr-cahce with global photon':
                var photon_count = tool.getParameter("photon count");
                SC_FILE.writeln("\tsamples "+gi_samples);
                SC_FILE.writeln("\ttolerance "+gi_error);
                SC_FILE.writeln("\tspacing "+gi_min+" "+gi_max);
                SC_FILE.writeln("\tglobal "+photon_count+" grid 100 0.5");
                if (!photonSwc) OS.messageBox("this gi setting requires photons.","some gi settings requires photons. Exporter adds photons automatically.");
                photonSwc = 1;
                break;
        }
        SC_FILE.writeln("}");
        
    }
    
    if (photonSwc) {
        var photon_count = tool.getParameter("photon count");
        var photon_type = Photon_Types[tool.getParameter("photon type")];
        SC_FILE.writeln("photons {\n\t "+photon_type+" "+photon_count+" kd 100 0.5\n}\n");
    }

}

function writeSetting(tool) {
    var doc = tool.document();
    var cam = doc.activeCamera();
    
    var img_width = cam.getParameter("resolutionX");
    var img_height = cam.getParameter("resolutionY");
    var amin = tool.getParameter("Min AA");
    var amax = tool.getParameter("Max AA");
    var asamples = tool.getParameter("AA samples");
    var ajitter = (tool.getParameter("AA jitter"))? 'true' : 'false';
    
    var ac_type = Accel_list[tool.getParameter("Accelerator Type")];
    var buc_size = tool.getParameter("Bucket Size");
    var buc_type = Bucket_list[tool.getParameter("Bucket Order")];
    var fil_type = ImageFilter_list[tool.getParameter("Image Filter")];
    
    SC_FILE.writeln("image {");
    SC_FILE.writeln("\tresolution "+img_width+" "+img_height);
    SC_FILE.writeln("\taa "+amin+" "+amax);
    SC_FILE.writeln("\tsamples "+asamples);
    SC_FILE.writeln("\tfilter "+fil_type);
    SC_FILE.writeln("\tjitter "+ajitter);
    SC_FILE.writeln("}\n");
    
    //SC_FILE.writeln("accel "+ac_type); // 
    SC_FILE.writeln("bucket "+buc_size+" "+buc_type);
    
    var diff = tool.getParameter("diffuse");
    var refl = tool.getParameter("reflection");
    var refr = tool.getParameter("refraction");
    
    SC_FILE.writeln("\ntrace-depths {");
    SC_FILE.writeln("\tdiff "+diff);
    SC_FILE.writeln("\trefl "+refl);
    SC_FILE.writeln("\trefr "+refr);    
    SC_FILE.writeln("}");
}

function writeLight(light_data, tool) {
    var light = light_data[0];
    if (light.family() == LIGHTFAMILY) {
        var i,j;
        var light_name = lightName(light.getParameter("name"));
        var light_mat = light.obj2WorldMatrix();
        var pos = new Vec3D(light_mat.m03, light_mat.m13, light_mat.m23); // from the matrix of object.
        var rot = light_data[1];
        var scale = light_data[2];
        var light_type = light.getParameter("lightType");
        var light_intensity = light.getParameter("intensity");
        var light_color = light.getParameter("color");
        var light_atten = light.getParameter("attenuation");
        //light_color = light_color.multiply(light_intensity);
        switch(parseInt(light_type)) {
            case 3: // Point
                SC_FILE.writeln("light {");
                SC_FILE.writeln("\ttype point");
                SC_FILE.writeln("\tcolor "+vector3Line(light_color));
                SC_FILE.writeln("\tpower "+light_intensity);
                SC_FILE.writeln("\tp "+vector3Line(pos));
                SC_FILE.writeln("}");
                break;
            case 1: // Area
                var axis = light.getParameter("axis");
                var samples = light.getParameter("samples");
                var light_width = light.getParameter("width");
                var light_height = light.getParameter("height");
                SC_FILE.writeln("light {");
                SC_FILE.writeln("\ttype meshlight");
                SC_FILE.writeln("\tname \""+light_name+"\"");
                SC_FILE.writeln("\temit "+colorLine(light_color));
                SC_FILE.writeln("\tradiance "+light_intensity);
                SC_FILE.writeln("\tsamples "+samples);
                SC_FILE.writeln("\tpoints 4");
                var p0 = new Vec3D(light_width / -2,  0, light_height / 2);
                var p1 = new Vec3D(light_height / 2,  0, light_height / 2);
                var p2 = new Vec3D(light_width / 2, 0, light_height / -2);
                var p3 = new Vec3D(light_width / -2, 0, light_height / -2);
                SC_FILE.writeln("\t\t"+vector3Line(light_mat.multiply(p0)));
                SC_FILE.writeln("\t\t"+vector3Line(light_mat.multiply(p1)));
                SC_FILE.writeln("\t\t"+vector3Line(light_mat.multiply(p2)));
                SC_FILE.writeln("\t\t"+vector3Line(light_mat.multiply(p3)));
                SC_FILE.writeln("\ttriangles 2");
                SC_FILE.writeln("\t\t2 1 0");
                SC_FILE.writeln("\t\t3 2 0");
                SC_FILE.writeln("}");
                break;
            case 4: // Spot
                var light_angle = light.getParameter("cutOffAngle");
                light_angle = light_angle * Math.PI / 180; // to radian
                var radius = Math.sin(light_angle) * 2;
                var light_target = light_mat.multiply(new Vec3D(0, -1, 0));
                SC_FILE.writeln("light {");
                SC_FILE.writeln("\ttype directional");
                SC_FILE.writeln("\tsource "+vector3Line(pos));
                SC_FILE.writeln("\ttarget "+vector3Line(light_target));
                SC_FILE.writeln("\tradius "+radius);
                SC_FILE.writeln("\temit "+colorLine(light_color));
                SC_FILE.writeln("\tintensity "+light_intensity);
                SC_FILE.writeln("}");
                break;
        }
    }
}

function writePrimitive(obj_data, tool) {
    var obj = obj_data[0];
    if (obj.family() == NGONFAMILY) {
        var shape = obj.type();
        var i,j;
        var obj_name = shapeName(obj.getParameter("name"));
        var obj_info = obj.getParameter("name");
        var obj_infos = obj_info.split("_");
        var obj_mat = obj.obj2WorldMatrix(); // this affects only position parameter. 
        var pos = new Vec3D(obj_mat.m03, obj_mat.m13, obj_mat.m23); //from the matrix of object.
        var rot = obj_data[1]; //
        var scale = obj_data[2]; //
        var obj_matlist = obj.materialTags();
        var obj_matsels = new Array();
        var obj_accel = Accel_list[tool.getParameter("Accelerator Type")];
        //
        var mod_name = undefined;
        if (obj_matlist.length < 1) {
            obj_matsels.push(new Array(0, DEF_MAT_NAME));
        } else {
            for (i = 0;i < obj_matlist.length; i++) {
                var mat_tag = obj_matlist[i];
                var doc = mat_tag.document();
                var material = doc.materialAtIndex(mat_tag.linkedToMaterial());
                var mat_type = material.getParameter("shaderType");
                if (mat_type == "Material") {
                    var bumpT = material.getParameter("bumpTex");
                    var bumpIntensity = material.getParameter("bumpIntensity");
                    if (bumpT != 'none') {
                        mod_name = writeModifier(SC_FILE, 'bump', bumpT, bumpIntensity);
                    }
                }
                obj_matsels.push(new Array(0, MAT_LIST[mat_tag.linkedToMaterial()]));
            }
        }
        SC_FILE.writeln("object {");
        if (obj_matsels.length == 1) {
            SC_FILE.writeln("\tshader \""+obj_matsels[0][1]+"\"");
        } else {
            SC_FILE.writeln("\tshaders "+obj_matsels.length);
            for (i = 0;i < obj_matsels.length;i++) {
                SC_FILE.writeln("\t\t\""+obj_matsels[i][1]+"\"");
            }
        }
        if (mod_name) {
            SC_FILE.writeln("\tmodifier \""+mod_name+"\"");
        }
        switch(shape) {
            case BALL:
                var radius = obj.getParameter("radius");
                scale = scale.multiply(radius);
                writePrimitiveMatrix(pos, scale, rot, "rotatex -90");
        if (obj_accel != 'none') SC_FILE.writeln("\taccel "+obj_accel);
                SC_FILE.writeln("\ttype sphere");
                SC_FILE.writeln("\tname \""+obj_name+"\"");
                break;
            case CONE: // for 
                var radius = obj.getParameter("radius");
                var b_type = 'sphere';
                if (obj_infos.length > 1) {
                    var b_type = obj_infos[1];
                }
                switch(b_type) {
                    case 'teapot':
                        scale = scale.multiply(radius*0.012);
                        writePrimitiveMatrix(pos, scale, rot, "rotatex -90");
                        if (obj_accel != 'none') SC_FILE.writeln("\taccel "+obj_accel);
                        SC_FILE.writeln("\ttype teapot");
                        SC_FILE.writeln("\tname \""+obj_name+"\"");
                        var subdiv = 2;
                        var smooth = 'true';
                        if (obj_infos.length > 2) subdiv = parseInt(obj_infos[2]);
                        SC_FILE.writeln("\tsubdivs "+subdiv);
                        if (obj_infos.length > 3) smooth = obj_infos[3];
                        SC_FILE.writeln("\tsmooth "+smooth);
                        
                        break;
                    case 'gumbo':
                        scale = scale.multiply(radius*0.1);
                        writePrimitiveMatrix(pos, scale, rot, "translate -15 -4 0\n\trotatex -90\n\trotatey 180");
                        if (obj_accel != 'none') SC_FILE.writeln("\taccel "+obj_accel);
                        SC_FILE.writeln("\ttype gumbo");
                        SC_FILE.writeln("\tname \""+obj_name+"\"");
                        var subdiv = 2;
                        var smooth = 'true';
                        if (obj_infos.length > 2) subdiv = parseInt(obj_infos[2]);
                        SC_FILE.writeln("\tsubdivs "+subdiv);
                        if (obj_infos.length > 3) smooth = obj_infos[3];
                        SC_FILE.writeln("\tsmooth "+smooth);
                        
                        break;
                    case 'sphere':
                    default:
                        writePrimitiveMatrix(pos, scale, rot, "rotatex -90");
                        if (obj_accel != 'none') SC_FILE.writeln("\taccel "+obj_accel);
                        SC_FILE.writeln("\ttype sphere");
                }
                break;
            case PLANE: // plane is infinitive!!
                var ori = obj.getParameter("orientation");
                var width = obj.getParameter("sizeX");
                var height = obj.getParameter("sizeY");
                //if (obj_accel != 'none') SC_FILE.writeln("\taccel "+obj_accel); // seems can not use accel for infinite plane?
                SC_FILE.writeln("\ttype plane");
                SC_FILE.writeln("\tname \""+obj_name+"\"");
                SC_FILE.writeln("\tp "+vector3Line(new Vec3D(0,0,0)));
                SC_FILE.writeln("\tn "+vector3Line(Normal_List[ori]));
                break;
            case TORUS:
                var radius_ring = obj.getParameter("radiusRing");
                var radius_tube = obj.getParameter("radiusTube");
                //rot.y -= 90;
                writePrimitiveMatrix(pos, scale, rot, "rotatex 90");
                if (obj_accel != 'none') SC_FILE.writeln("\taccel "+obj_accel);
                SC_FILE.writeln("\ttype torus");
                SC_FILE.writeln("\tname \""+obj_name+"\"");
                SC_FILE.writeln("\tr "+radius_tube.toPrecision(4)+" "
                    +radius_ring.toPrecision(4));
                break;
        }
        SC_FILE.writeln("}");
        
        return obj_name;
    }
}

function writePrimitiveMatrix(pos, scale, rot, preDef) {
        SC_FILE.writeln("\ttransform {");
        if (preDef) SC_FILE.writeln("\t"+preDef);
        SC_FILE.writeln("\tscale "+vector3Line(scale));
        if (rot.y != 0) SC_FILE.writeln("\trotatex "+rot.y.toPrecision(6));
        if (rot.z != 0) SC_FILE.writeln("\trotatez "+rot.z.toPrecision(6));
        if (rot.x != 0) SC_FILE.writeln("\trotatey "+rot.x.toPrecision(6));
        if (pos.norm() != 0) SC_FILE.writeln("\ttranslate "+vector3Line(pos));
        SC_FILE.writeln("\t}");
}

function writeMatrix(file, pos, scale, rot) {
        file.writeln("\ttransform {");
        file.writeln("\tscale "+vector3Line(scale));

        if (rot.y != 0) file.writeln("\trotatex "+rot.y.toPrecision(6));
        if (rot.z != 0) file.writeln("\trotatez "+rot.z.toPrecision(6));
        if (rot.x != 0) file.writeln("\trotatey "+rot.x.toPrecision(6));
        if (pos.norm() != 0) file.writeln("\ttranslate "+vector3Line(pos));
        file.writeln("\t}");
}

function writeModifier(file, type, tex, intensity) {
    var mod_name = modName(type, filenameForTexture(tex));
    
    file.writeln("modifier {");
    file.writeln("\tname \""+mod_name+"\"");
    file.writeln("\ttype "+type);
    file.writeln("\ttexture \""+filenameForTexture(tex)+"\"");
    file.writeln("\tscale "+intensity);
    file.writeln("}");
    
    return mod_name;
}

function writeMesh(obj_data, tool) {
    var obj = obj_data[0];
    if (obj.family() == NGONFAMILY) {
        var i,j;
        var obj_name = shapeName(obj.getParameter("name"));
        var obj_scriptName = obj.getParameter("scriptName");
        var obj_mat = obj.obj2WorldMatrix();
        var obj_matR = new Mat4D(ROTATE_HPB, obj_data[1].h, obj_data[1].p, obj_data[1].b);
        var obj_matS = new Mat4D(SCALE, obj_data[2].x, obj_data[2].y, obj_data[2].z);
        var obj_matlist = obj.materialTags();
        var obj_matsels = new Array;
        var core = obj.modCore();
        var vertCount = core.vertexCount();
        var polyCount = core.polygonCount();
        var triCount = core.triangleCount();
        var obj_infos = (obj.getParameter("name")).split("_");
        var obj_type = (obj_infos.length > 1)? obj_infos[1] : 'def';
        var obj_accel = Accel_list[tool.getParameter("Accelerator Type")];
        var obj_pos = new Vec3D(obj_mat.m03, obj_mat.m13, obj_mat.m23);

        // polygon check
        if (triCount < 1) {
            var res = OS.messageBox("Polygon Alert - "+obj.getParameter("name"),
                "this object has no polygon. it may cause some error.\nplease check a object.");
        }
        if (tool.getParameter("separate mesh file")) {
            var dir = SC_FILE.directory();
            var file_name = SC_FILE.lastPathComponent();
            var file_name_list = file_name.split(".");
            var mesh_filename = file_name_list[0] + '_' + obj_name + '.geo.sc';
            var path = dir + '/' + mesh_filename;
            var file = new File(path);
            file.open(WRITE_MODE);
        } else {
            file = SC_FILE;
        }
        
        var mod_name = undefined;
        if (obj_matlist.length < 1) {
            obj_matsels.push(new Array(0, DEF_MAT_NAME));
        } else {
            for (i = 0;i < obj_matlist.length; i++) {
                var mat_tag = obj_matlist[i];
                var sel = mat_tag.getParameter("shadeSelection");
                var doc = mat_tag.document();
                var material = doc.materialAtIndex(mat_tag.linkedToMaterial());
                var mat_type = material.getParameter("shaderType");
                if (mat_type == "Material") {
                    var bumpT = material.getParameter("bumpTex");
                    var bumpIntensity = material.getParameter("bumpIntensity");
                    if (bumpT != 'none') {
                        mod_name = writeModifier(file, 'bump', bumpT, bumpIntensity);
                    }
                }
                obj_matsels.push(new Array(sel, MAT_LIST[mat_tag.linkedToMaterial()]));
            }
        }
        if (obj_type == 'light') { // mesh light
            if (obj_matlist.length > 0) {
                var mat_tag = obj_matlist[0];
                var material = doc.materialAtIndex(mat_tag.linkedToMaterial());
                var mat_type = material.getParameter("shaderType");
            } else var mat_type = '';
            var emit_color = new Vec3D(1, 1, 1);
            var emit_intensity = 1;
            var emit_samples = 1;
            if (mat_type == 'Material') {
                emit_color = material.getParameter("colorColor");
            } else if (mat_type == 'SolidColor') {
                emit_color = material.getParameter("colorColor");
                emit_intensity = material.getParameter("colorIntensity");            
            }
            emit_samples= (obj_infos.length > 2)? obj_infos[2] : 1;
            file.writeln("light {");
            file.writeln("\ttype meshlight");
            file.writeln("\tname \""+obj_name+"\"");
            file.writeln("\temit "+colorLine(emit_color));
            file.writeln("\tradiance "+emit_intensity);
            file.writeln("\tsamples "+emit_samples);
            // vertices
            file.writeln("\tpoints "+vertCount);
            for (i = 0;i < vertCount;i++) {
                var vert = obj_mat.multiply(core.vertex(i));
                file.writeln("\t\t"+vector3Line(vert));
            }
            // face list
            file.writeln("\ttriangles "+triCount);
            for (i = 0;i < polyCount;i++) {
                var polySize = core.polygonSize(i);
                if (polySize > 3) {
                    for (j = 0;j < polySize-2;j++) {
                        var nums = core.triangle(i,j);
                        var n1 = core.vertexIndex(i,nums[0]);
                        var n2 = core.vertexIndex(i,nums[2]);
                        var n3 = core.vertexIndex(i,nums[1]);
                        file.writeln("\t\t"+n1+" "+n2+" "+n3);
                    }
                } else {
                    var n1 = core.vertexIndex(i,0);
                    var n2 = core.vertexIndex(i,2);
                    var n3 = core.vertexIndex(i,1);
                    file.writeln("\t\t"+n1+" "+n2+" "+n3);
                }
            }
        } else {
            // shape header.
            file.writeln("object {");
            if (obj_matsels.length == 1) {
                file.writeln("\tshader \""+obj_matsels[0][1]+"\"");
            } else {
                file.writeln("\tshaders "+obj_matsels.length);
                for (i = 0;i < obj_matsels.length;i++) {
                    file.writeln("\t\t\""+obj_matsels[i][1]+"\"");
                }
            }
            if (mod_name) {
                file.writeln("\tmodifier \""+mod_name+"\"");
            }
            if (obj_accel != 'none') {
                file.writeln("\taccel "+obj_accel);
            }
            
            if (obj_scriptName == 'Hair.js') {
                var h_segments = obj.getParameter("segments");
                var h_width = obj.getParameter("hair width");
                var h_length = obj.getParameter("hair length");
                var curlX = obj.getParameter("curly x");
                var curlY = obj.getParameter("curly y");
                var curlZ = obj.getParameter("curly z");
                var randCurly = obj.getParameter("random curly");
                var gravX = obj.getParameter("direction x");
                var gravY = obj.getParameter("direction y");
                var gravZ = obj.getParameter("direction z");
                var randA = obj.getParameter("random a");
                var randB = obj.getParameter("random b");
                var gravGV = new Vec3D(gravX, gravY, gravZ);
                var curlV = new Vec3D(curlX, curlY, curlZ);
                
                if (obj.childCount() > 0) {
                    var g_obj = obj.childAtIndex(0);
                    if (g_obj.family() == NGONFAMILY) {
                        //
                        var g_core = g_obj.modCore();
                        
                        var g_polyCount = g_core.polygonCount();
                        var g_mat = g_obj.objMatrix();
                        
                        file.writeln("\ttype hair");
                        file.writeln("\tname \""+obj_name+"\"");
                        file.writeln("\tsegments "+h_segments);
                        file.writeln("\twidth "+h_width);
                        file.writeln("\tpoints "+g_polyCount*3*(h_segments+1));
                        for (i = 0;i < g_polyCount;i++) {
                            var gSize = g_core.polygonSize(i);
                            var rand1 = Math.random();
                            var rand2 = Math.random();
                            var rand3 = Math.random();
                            var randA_h = randA / 2;
                            var curlX_h = curlX / 2;
                            var curlY_h = curlY / 2;
                            var curlZ_h = curlZ / 2;
                                    
                            var gravV = gravGV.add(new Vec3D(rand1*randA-randA_h,rand2*randA-randA_h,rand3*randA-randA_h));
                            var vert = new Vec3D(0,0,0);
                            for (j = 0;j < gSize;j++) {
                                var pvert = g_core.vertex(g_core.vertexIndex(i,j));
                                var puv = g_core.uvCoord(i,j);
                                vert = vert.add(pvert);
                            }
                            vert = vert.multiply(1/gSize);
                            
                            var normal = g_core.normal(i);
        
                            normal = normal.multiply(h_length);
                            normal = vert.add(normal);
                            if (randCurly) curlV = new Vec3D(rand1*curlX - curlX_h, rand2*curlY - curlY_h, rand3*curlZ - curlZ_h);
                            
                            vert = g_mat.multiply(vert);
                            normal = g_mat.multiply(normal);
                            
                            var k;
                            var segV = normal.sub(vert);
                            segV = segV.multiply(1/h_segments);
                            var nextV = vert;
                            var randB_h = randB / 2;
                            file.writeln("\t\t"+vector3Line(vert));
                            for (k = 0;k < h_segments;k++) {
                                var segFac = (k*2+1)/(h_segments*h_segments);
                                var nextCurlV = curlV.multiply(segFac);
                                var curlMat = new Mat4D(ROTATE, nextCurlV.x, nextCurlV.y, nextCurlV.z);
                                nextV = nextV.add(segV.add(new Vec3D(Math.random()*randB-randB_h,Math.random()*randB-randB_h,Math.random()*randB-randB_h)));
                                nextV = nextV.add(gravV.multiply(segFac));
                                nextV = vert.add(curlMat.multiply(nextV.sub(vert)));
                                file.writeln("\t\t"+vector3Line(nextV));
                            }
                        }  
                    }
                }
            } else if (obj_type == 'hair') {
                file.writeln("\ttype hair");
                file.writeln("\tname \""+obj_name+"\"");
                var h_segments = 1;
                var h_width = 0.01;
                if (obj_infos.length > 2) h_segments = parseInt(obj_infos[2]);
                if (obj_infos.length > 3) h_width = parseFloat(obj_infos[3]);
                file.writeln("\tsegments "+h_segments);
                file.writeln("\twidth "+h_width);
                file.writeln("\tpoints "+polyCount*3*(h_segments+1));
                for (i = 0;i < polyCount;i++) {
                    var polySize = core.polygonSize(i);
                    var norm = obj_mat.multiply(core.normal(i));
                    var vert = new Vec3D(0,0,0);
                    for (j = 0;j < polySize;j++) {
                        vert = vert.add(core.vertex(core.vertexIndex(i,j)));
                    }
                    vert = obj_mat.multiply(vert.multiply(1/polySize));
                    file.writeln("\t\t"+vector3Line(vert));
                    file.writeln("\t\t"+vector3Line(norm));
                    for (j = 1;j < h_segments;j++) {
                        var trans_mat = new Mat4D(TRANSLATE, Math.random()*0.2-0.1, Math.random()*0.2-0.1, Math.random()*0.2-0.1);
                        var scale_mat = new Mat4D(SCALE, 1.1, 1.1, 1.1);
                        norm = trans_mat.multiply(scale_mat.multiply(norm));
                        file.writeln("\t\t"+vector3Line(norm));
                    }
                }
            } else if (obj_type == 'particles') {
                file.writeln("\ttype particles");
                file.writeln("\tname \""+obj_name+"\"");
                var par_filename = obj_name + ".dat";
                var par_dir = file.directory();
                var par_radius = 0.01;
                if (obj_infos.length > 2) par_radius = parseFloat(obj_infos[2]);
                file.writeln("\tfilename \""+par_dir+"/"+par_filename+"\"");
                file.writeln("\tradius "+par_radius);
                var par_file = new File(par_dir+"/"+par_filename);
                par_file.open(WRITE_MODE);
                for (i = 0;i < vertCount;i++) {
                    var vert = obj_mat.multiply(core.vertex(i));
                    par_file.writeFloat(vert.x);
                    par_file.writeFloat(vert.y);
                    par_file.writeFloat(vert.z);
                }
                par_file.close();
            } else {
                //writeMatrix(file, obj_pos, obj_scale, obj_rot);
                file.writeln("\ttype generic-mesh");
                file.writeln("\tname \""+obj_name+"\"");
                // vertex list
                file.writeln("\tpoints "+vertCount);
                for (i = 0;i < vertCount;i++) {
                    var vert = obj_mat.multiply(core.vertex(i));
                    file.writeln("\t\t"+vector3Line(vert));
                }
                // face list
                file.writeln("\ttriangles "+triCount);
                for (i = 0;i < polyCount;i++) {
                    var polySize = core.polygonSize(i);
                    if (polySize > 3) {
                        for (j = 0;j < polySize-2;j++) {
                            var nums = core.triangle(i,j);
                            var n1 = core.vertexIndex(i,nums[0]);
                            var n2 = core.vertexIndex(i,nums[2]);
                            var n3 = core.vertexIndex(i,nums[1]);
                            file.writeln("\t\t"+n1+" "+n2+" "+n3);
                        }
                    } else {
                        var n1 = core.vertexIndex(i,0);
                        var n2 = core.vertexIndex(i,2);
                        var n3 = core.vertexIndex(i,1);
                        file.writeln("\t\t"+n1+" "+n2+" "+n3);
                    }
                }
                // face normal
                file.writeln("\tnormals facevarying");
                for (i = 0;i < polyCount;i++) {
                    var polySize = core.polygonSize(i);
                    if (polySize > 3) {
                        for (j = 0;j < polySize-2;j++) {
                            var nums = core.triangle(i,j);
                            var norm1 = obj_matS.multiply(obj_matR.multiply(core.normal(i,nums[0])));
                            var norm2 = obj_matS.multiply(obj_matR.multiply(core.normal(i,nums[2])));
                            var norm3 = obj_matS.multiply(obj_matR.multiply(core.normal(i,nums[1])));
                            file.writeln("\t\t"+vector3Line(norm1)+" "
                                        +vector3Line(norm2)+" "
                                        +vector3Line(norm3));
                        }
                    } else {
                        var norm1 = obj_matS.multiply(obj_matR.multiply(core.normal(i,0)));
                        var norm2 = obj_matS.multiply(obj_matR.multiply(core.normal(i,2)));
                        var norm3 = obj_matS.multiply(obj_matR.multiply(core.normal(i,1)));
                        file.writeln("\t\t"+vector3Line(norm1)+" "
                                        +vector3Line(norm2)+" "
                                        +vector3Line(norm3));
                    }
                }
                // face uv
                file.writeln("\tuvs facevarying");
                for (i = 0;i < polyCount;i++) {
                    var polySize = core.polygonSize(i);
                    for (j = 0;j < polySize-2;j++) {
                        var nums = core.triangle(i,j);
                        var uv1 = core.uvCoord(i,nums[0]);
                        var uv2 = core.uvCoord(i,nums[2]);
                        var uv3 = core.uvCoord(i,nums[1]);
                        uv1.y = Math.abs(uv1.y-1);
                        uv2.y = Math.abs(uv2.y-1);
                        uv3.y = Math.abs(uv3.y-1);
                        file.writeln("\t\t"+uv1.x.toFixed(6)+" "+uv1.y.toFixed(6)
                                        +" "+uv2.x.toFixed(6)+" "+uv2.y.toFixed(6)
                                        +" "+uv3.x.toFixed(6)+" "+uv3.y.toFixed(6));
                    }
                }
                // polygon selection
                if (obj_matsels.length > 1) {
                    file.writeln("\tface_shaders");
                    for (i = 0;i < polyCount;i++) {
                        var polySize = core.polygonSize(i);
                        var mat_num = 0;
                        for (j = 0;j < obj_matsels.length;j++) {
                            var matsel = obj_matsels[j][0] - 1;
                            if (matsel < 0) { // all
                                mat_num = 0;
                            } else { // check selcted
                                core.setActivePolygonSelection(matsel);
                                if (core.polygonSelection(i)) mat_num = j;
                            }
                        }
                        core.setActivePolygonSelection(0);
                        if (polySize > 3) {
                            for (j = 0;j < polySize-2;j++) {
                                file.writeln("\t\t"+mat_num);
                            }
                        } else {
                            file.writeln("\t\t"+mat_num);
                        }
                    }
                }
            }
        }
        // footer
        file.writeln("}\n");
        
        if (tool.getParameter("separate mesh file")) {
            file.close();
            SC_FILE.writeln("## "+obj.getParameter("name")+" as \""+obj_name+"\"");
            SC_FILE.writeln("include \""+mesh_filename+"\"");
        }
        
        return obj_name;
    }
}

function writeMaterial(material, info, mode) {
    var mat_name = material.getParameter("name");
    var mat_type = material.getParameter("shaderType");
    var matrix_line = '';
    var mat_infos = mat_name.split("_");
    if (mat_infos.length < 2) mat_infos[1] = 'auto';
    //
    switch (mat_type) {
        case "Material":
            var colorC = material.getParameter("colorColor");
            var colorT = material.getParameter("colorTex");
            var defIntensity = material.getParameter("diffIntensity");
            var colorMix = material.getParameter("colorMix");
            var specC = material.getParameter("specularColor");
            var specT = material.getParameter("specTex");
            var specMix = material.getParameter("specMix");
            var specSize = material.getParameter("specSize");
            var refC = material.getParameter("reflColor");
            var refT = material.getParameter("reflTex");
            var refIntensity = material.getParameter("reflIntensity");
            var refMix = material.getParameter("reflMix");
            var emitC = material.getParameter("emissionColor");
            var transC = material.getParameter("transColor");
            var transT = material.getParameter("transTex");
            var transIntensity = material.getParameter("transIntensity");
            var transMix = material.getParameter("transMix");
            var fresnel = material.getParameter("transFresnel");
            var ior = material.getParameter("transEta");

            if (!colorT) colorMix = 1;
            if (!specT) specMix = 1;
            if (!refT) refMix = 1;
            if (!transT) transMix = 1;
            
            // diffuse calculation
            colorC = colorC.multiply(defIntensity);
            // specular calculation
            specSize = (128 - specSize) / 128 * 1000; // 0-1000
            //
            mat_name = matName('mat',mat_name);
            SC_FILE.writeln("shader {");
            SC_FILE.writeln("\tname \""+mat_name+"\"");
            switch(mat_infos[1]) {
                case 'diffuse':
                    SC_FILE.writeln("\ttype diffuse");
                    if (colorT != 'none') SC_FILE.writeln("\ttexture \""+filenameForTexture(colorT)+"\"");
                    else SC_FILE.writeln("\tdiff "+colorLine(colorC));
                    break;
                case 'phong':
                    SC_FILE.writeln("\ttype phong");
                    if (colorT != 'none') SC_FILE.writeln("\ttexture \""+filenameForTexture(colorT)+"\"");
                    else SC_FILE.writeln("\tdiff "+colorLine(colorC));
                    SC_FILE.writeln("\tspec "+colorLine(specC)+" "+specSize);
                    var spec_samples = 4;
                    if (mat_infos.length > 2) {
                        spec_samples = parseInt(mat_infos[2]);
                        spec_samples = (spec_samples > 0)? spec_samples : 4; // default 4;
                    }
                    SC_FILE.writeln("\tsamples "+spec_samples);
                    break;
                case 'amc-occ':
                    SC_FILE.writeln("\ttype amb-occ");
                    if (colorT != 'none') SC_FILE.writeln("\ttexture \""+filenameForTexture(colorT)+"\"");
                    else SC_FILE.writeln("\tdiff "+colorLine(colorC));
                    SC_FILE.writeln("\tdark "+colorLine(emitC));
                    var am_samples = 4;
                    var am_dist = 6;
                    if (mat_infos.length > 3) {
                        am_samples = parseInt(mat_infos[2]);
                        am_dist = parseFloat(mat_infos[3]);
                        am_samples = (am_samples > 0)? am_samples : 4; // default 4;
                        am_dist = (am_dist > 0)? am_dist : 6; // default 6;
                    }
                    SC_FILE.writeln("\tsamples "+am_samples);
                    SC_FILE.writeln("\tdist "+am_dist);                    
                    break;
                case 'mirror':
                    SC_FILE.writeln("\ttype mirror");
                    SC_FILE.writeln("\trefl "+vector3Line(refC.multiply(refIntensity)));
                    break;
                case 'glass':
                    SC_FILE.writeln("\ttype glass");
                    SC_FILE.writeln("\teta "+ior);
                    SC_FILE.writeln("\tcolor "+colorLine(transC.multiply(transIntensity)));
                    if (mat_infos.length > 2) {
                        var ab_dist = parseFloat(mat_infos[2]);
                        ab_dist = (ab_dist > 0)? ab_dist : 1; // default 1;
                        SC_FILE.writeln("\tabsorbtion.distance "+ab_dist);
                        SC_FILE.writeln("\tabsorbtion.color "+colorLine(colorC));
                    }
                    break;
                case 'shiny':
                    SC_FILE.writeln("\ttype shiny");
                    if (colorT != 'none') SC_FILE.writeln("\ttexture \""+filenameForTexture(colorT)+"\"");
                    else SC_FILE.writeln("\tdiff "+colorLine(colorC));
                    var sh_ref = refIntensity;
                    if (mat_infos.length > 2) {
                        sh_ref = parseFloat(mat_infos[2]);
                        sh_ref = (sh_ref > 0)? sh_ref : refIntensity;
                    }
                    SC_FILE.writeln("\trefl "+sh_ref);
                    break;
                case 'ward':
                    SC_FILE.writeln("\ttype ward");
                    if (colorT != 'none') SC_FILE.writeln("\ttexture \""+filenameForTexture(colorT)+"\"");
                    else SC_FILE.writeln("\tdiff "+colorLine(colorC));
                    SC_FILE.writeln("\tspec "+colorLine(specC));
                    var roughX = 0.5;
                    var roughY = 0.5;
                    if (mat_infos.length > 3) {
                        roughX = parseFloat(mat_infos[2]);
                        roughY = parseFloat(mat_infos[3]);
                        roughX = (roughX > 0)? roughX : 0.5;
                        roughY = (roughY > 0)? roughY : 0.5;
                    }
                    SC_FILE.writeln("\trough "+roughX+" "+roughY);
                    var samples = 2;
                    if (mat_infos.length > 4) {
                        samples = parseInt(mat_infos[4]);
                        samples = (samples > 0)? samples : 2;
                    }
                    SC_FILE.writeln("\tsamples "+samples);
                    break;
                case 'view-caustics':
                case 'view-irradiance':
                case 'view-global':
                case 'id':
                    SC_FILE.writeln("\ttype "+mat_infos[1]);
                    break;
                case 'uber':
                    SC_FILE.writeln("\ttype "+mat_infos[1]);
                    SC_FILE.writeln("\tdiff "+colorLine(colorC));
                    if (colorT != 'none') SC_FILE.writeln("\tdiff.texture \""+filenameForTexture(colorT)+"\"");
                    SC_FILE.writeln("\tdiff.blend "+colorMix);
                    SC_FILE.writeln("\tspec "+colorLine(specC));
                    if (specT != 'none') SC_FILE.writeln("\tspec.texture \""+filenameForTexture(specT)+"\"");
                    SC_FILE.writeln("\tspec.blend "+specMix);
                    var glossy = 0.5;
                    if (mat_infos.length > 2) {
                        glossy = parseFloat(mat_infos[2]);
                        glossy = (glossy > 0)? glossy : 0.5;
                    }
                    SC_FILE.writeln("\tglossy "+glossy);
                    var samples = 2;
                    if (mat_infos.length > 3) {
                        samples = parseInt(mat_infos[3]);
                        samples = (samples > 0)? samples : 2;
                    }
                    SC_FILE.writeln("\tsamples "+samples);
                    break;
                case 'auto':
                default:
                    if (transIntensity > 0) { // 1. transparency for glass
                        SC_FILE.writeln("\ttype glass");
                        SC_FILE.writeln("\teta "+ior);
                        SC_FILE.writeln("\tcolor "+colorLine(transC.multiply(transIntensity)));
                    } else if (refIntensity > 0) { // 2. reflection for mirror
                        SC_FILE.writeln("\ttype mirror");
                        SC_FILE.writeln("\trefl "+colorLine(refC.multiply(refIntensity)));
                    } else { // 3. diffuse
                        SC_FILE.writeln("\ttype diffuse");
                        if (colorT != 'none') SC_FILE.writeln("\ttexture \""+filenameForTexture(colorT)+"\"");
                        else SC_FILE.writeln("\tdiff "+colorLine(colorC));
                    }
            }
            //            
            SC_FILE.writeln("}\n");
            break;
        case "SolidColor":
            var colorC = material.getParameter("colorColor");
            var colorIntensity = material.getParameter("colorIntensity");
            colorC = colorC.multiply(colorIntensity);
            mat_name = matName('const',mat_name);
            SC_FILE.writeln("shader {");
            SC_FILE.writeln("\tname \""+mat_name+"\"");
            SC_FILE.writeln("\ttype constant");
            SC_FILE.writeln("\tcolor "+vector3Line(colorC));
            
            SC_FILE.writeln("}\n");
            break;
        default: // dummy material.
            mat_name = matName('ud',mat_name);
            SC_FILE.writeln("shader {");
            SC_FILE.writeln("\tname \""+mat_name+"\"");
            SC_FILE.writeln("\ttype diffuse");
            SC_FILE.writeln("\tdiff 0.8 0.8 0.8");
            SC_FILE.writeln("}\n");
            break;
    }
    
    return mat_name;
}
// debug function.
function printVec3D(str, vec) {
    print(str + ':x:'+vec.x.toPrecision(3)+', y:'+vec.y.toPrecision(3)+', z:'+vec.z.toPrecision(3));
}
function printVec4D(vec) {
    print('r:g:b:a-'+vec.r+':'+vec.g+':'+vec.b+':'+vec.a);
}
function printMatrix(matrix) {
    print(matrix.m00.toPrecision(3)+':'+matrix.m01.toPrecision(3)+':'+matrix.m02.toPrecision(3)+':'+matrix.m03.toPrecision(3));
    print(matrix.m10.toPrecision(3)+':'+matrix.m11.toPrecision(3)+':'+matrix.m12.toPrecision(3)+':'+matrix.m13.toPrecision(3));
    print(matrix.m20.toPrecision(3)+':'+matrix.m21.toPrecision(3)+':'+matrix.m22.toPrecision(3)+':'+matrix.m23.toPrecision(3));
    print(matrix.m30.toPrecision(3)+':'+matrix.m31.toPrecision(3)+':'+matrix.m32.toPrecision(3)+':'+matrix.m33.toPrecision(3));    
}
function printParameterInfo(obj) {
    var i;
    var objInfo = obj.parameterInfo();
    print("--- parameterInfo ---");
    for (i = 0;i < objInfo.length;i++) {
        print(objInfo[i][0]+"  ["+objInfo[i][1]+"]");
        print(obj.getParameter(objInfo[i][0]));
    }
}
